<?php

namespace App\Http\Controllers\Backend\Games;

use App\Models\Games\Game;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Games\GameCategory;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class GameCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $categories = GameCategory::withCount('games')->latest()->paginate(10);
        return view('admin.pages.games.game_categories.index', compact('categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.pages.games.game_categories.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:game_categories',
            'icon' => 'required|string|max:50',
            'description' => 'nullable|string',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        GameCategory::create([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'icon' => $request->icon,
            'description' => $request->description,
            'is_active' => $request->is_active ?? true
        ]);

        return redirect()->route('admin.pages.games.game-categories.index')
            ->with('success', 'Game category created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(GameCategory $gameCategory)
    {
        return view('admin.pages.games.game_categories.edit', compact('gameCategory'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, GameCategory $gameCategory)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:game_categories,name,' . $gameCategory->id,
            'icon' => 'required|string|max:50',
            'description' => 'nullable|string',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $gameCategory->update([
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'icon' => $request->icon,
            'description' => $request->description,
            'is_active' => $request->is_active ?? true
        ]);

        return redirect()->route('admin.games.categories.index')
            ->with('success', 'Game category updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(GameCategory $gameCategory)
    {
        // Check if category has games
        $gamesCount = Game::where('category_id', $gameCategory->id)->count();
        
        if ($gamesCount > 0) {
            return redirect()->route('admin.games.categories.index')
                ->with('error', 'Cannot delete category that has games. Please reassign or delete the games first.');
        }
        
        $gameCategory->delete();
        
        return redirect()->route('admin.games.categories.index')
            ->with('success', 'Game category deleted successfully.');
    }
}