<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\HelpCenterArticle;
use App\Models\HelpCenterTopic;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class HelpCenterArticleController extends Controller
{
    public function index(Request $request)
    {
        $topics = HelpCenterTopic::orderBy('display_order')->orderBy('title')->get();
        $selectedTopicId = $request->input('topic_id');

        $articlesQuery = HelpCenterArticle::query()
            ->with('topic')
            ->orderBy('display_order')
            ->orderBy('title');

        if ($selectedTopicId) {
            $articlesQuery->where('topic_id', $selectedTopicId);
        }

        $articles = $articlesQuery->paginate(15);

        return view('admin.pages.help_center.articles.index', compact('articles', 'topics', 'selectedTopicId'));
    }

    public function create()
    {
        $topics = HelpCenterTopic::orderBy('display_order')->orderBy('title')->get();

        return view('admin.pages.help_center.articles.create', compact('topics'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'topic_id' => 'required|exists:help_center_topics,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:help_center_articles,slug',
            'hero_image' => 'nullable|string|max:255',
            'summary' => 'nullable|string',
            'content' => 'required|string',
            'estimated_reading_time' => 'nullable|integer|min:1|max:240',
            'display_order' => 'nullable|integer|min:0',
            'is_featured' => 'sometimes|boolean',
            'is_published' => 'sometimes|boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        $validated['slug'] = $validated['slug'] ?? Str::slug($validated['title']);
        $validated['is_featured'] = $request->boolean('is_featured');
        $validated['is_published'] = $request->boolean('is_published', true);
        $validated['display_order'] = $validated['display_order'] ?? 0;

        HelpCenterArticle::create($validated);

        return redirect()
            ->route('admin.help-center.articles.index')
            ->with('success', 'Help center article created successfully.');
    }

    public function edit(HelpCenterArticle $article)
    {
        $topics = HelpCenterTopic::orderBy('display_order')->orderBy('title')->get();

        return view('admin.pages.help_center.articles.edit', compact('article', 'topics'));
    }

    public function update(Request $request, HelpCenterArticle $article)
    {
        $validated = $request->validate([
            'topic_id' => 'required|exists:help_center_topics,id',
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:help_center_articles,slug,' . $article->id,
            'hero_image' => 'nullable|string|max:255',
            'summary' => 'nullable|string',
            'content' => 'required|string',
            'estimated_reading_time' => 'nullable|integer|min:1|max:240',
            'display_order' => 'nullable|integer|min:0',
            'is_featured' => 'sometimes|boolean',
            'is_published' => 'sometimes|boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        $validated['slug'] = $validated['slug'] ?? Str::slug($validated['title']);
        $validated['is_featured'] = $request->boolean('is_featured');
        $validated['is_published'] = $request->boolean('is_published', true);
        $validated['display_order'] = $validated['display_order'] ?? 0;

        $article->update($validated);

        return redirect()
            ->route('admin.help-center.articles.index')
            ->with('success', 'Help center article updated successfully.');
    }

    public function destroy(HelpCenterArticle $article)
    {
        $article->delete();

        return redirect()
            ->route('admin.help-center.articles.index')
            ->with('success', 'Help center article deleted successfully.');
    }
}
