<?php
namespace App\Http\Controllers\Backend;

use App\Models\Poke;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;

class PokeController extends Controller
{
    /**
     * Display poke statistics
     */
    public function index()
    {
        $totalPokes = Poke::count();
        $todayPokes = Poke::whereDate('created_at', today())->count();
        $unreadPokes = Poke::where('is_read', false)->count();
       
        $topSenders = User::select('users.id', 'users.name', 'users.avatar')
            ->join('pokes', 'users.id', '=', 'pokes.sender_id')
            ->selectRaw('COUNT(*) as poke_count')
            ->groupBy('users.id', 'users.name', 'users.avatar')
            ->orderBy('poke_count', 'desc')
            ->limit(10)
            ->get();
           
        $topReceivers = User::select('users.id', 'users.name', 'users.avatar')
            ->join('pokes', 'users.id', '=', 'pokes.receiver_id')
            ->selectRaw('COUNT(*) as poke_count')
            ->groupBy('users.id', 'users.name', 'users.avatar')
            ->orderBy('poke_count', 'desc')
            ->limit(10)
            ->get();
           
        $pokesByDay = Poke::select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as total'))
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->limit(30)
            ->get();
           
        return view('admin.pages.pokes.index', compact(
            'totalPokes',
            'todayPokes',
            'unreadPokes',
            'topSenders',
            'topReceivers',
            'pokesByDay'
        ));
    }
   
    /**
     * List all pokes
     */
    public function list(Request $request)
    {
        $query = Poke::with(['sender', 'receiver']);
       
        // Apply filters
        if ($request->has('sender')) {
            $query->where('sender_id', $request->input('sender'));
        }
       
        if ($request->has('receiver')) {
            $query->where('receiver_id', $request->input('receiver'));
        }
       
        if ($request->has('read_status')) {
            $query->where('is_read', $request->input('read_status') == 'read');
        }
       
        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->input('date_from'));
        }
       
        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->input('date_to'));
        }
       
       //Sort   
        $sort = $request->input('sort', 'newest');
        switch ($sort) {
            case 'oldest':
                $query->orderBy('created_at', 'asc');
                break;
            default:
                $query->orderBy('created_at', 'desc');
                break;
        }
       
        $pokes = $query->paginate(50);
       
        return view('admin.pages.pokes.list', compact('pokes'));
    }
   
    /**
     * Delete a poke
     */
    public function delete($id)
    {
        $poke = Poke::findOrFail($id);
        $poke->delete();
       
        return redirect()->back()
            ->with('success', 'Poke deleted successfully.');
    }
}