<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\VerificationRequest;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class VerificationController extends Controller
{
   
    
    /**
     * Display a listing of verification requests.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $request = request();
        $status = $request->get('status', 'pending');
        
        $verificationRequests = VerificationRequest::with('user')
            ->when($status === 'pending', function ($query) {
                return $query->where('status', 'pending');
            })
            ->when($status === 'approved', function ($query) {
                return $query->where('status', 'approved');
            })
            ->when($status === 'rejected', function ($query) {
                return $query->where('status', 'rejected');
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);
        
        return view('admin.pages.verification.index', compact('verificationRequests', 'status'));
    }

    /**
     * Display the specified verification request.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $verificationRequest = VerificationRequest::with('user', 'reviewer')
            ->findOrFail($id);
        
        return view('admin.pages.verification.show', compact('verificationRequest'));
    }

    /**
     * Approve the specified verification request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function approve(Request $request, $id)
    {
        $verificationRequest = VerificationRequest::findOrFail($id);
        
        // Only allow approving pending requests
        if ($verificationRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'Only pending verification requests can be approved.');
        }
        
        // Update verification request status
        $verificationRequest->update([
            'status' => 'approved',
            'admin_notes' => $request->admin_notes,
            'reviewed_at' => now(),
            'reviewed_by' => Auth::id()
        ]);
        
        // Update user's verified status if needed
        // You could add a 'is_verified' column to users table if needed
        
        return redirect()->route('admin.verification.index')
            ->with('success', 'Verification request has been approved successfully.');
    }

    /**
     * Reject the specified verification request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function reject(Request $request, $id)
    {
        $request->validate([
            'admin_notes' => 'required|string|max:1000',
        ]);
        
        $verificationRequest = VerificationRequest::findOrFail($id);
        
        // Only allow rejecting pending requests
        if ($verificationRequest->status !== 'pending') {
            return redirect()->back()->with('error', 'Only pending verification requests can be rejected.');
        }
        
        // Update verification request status
        $verificationRequest->update([
            'status' => 'rejected',
            'admin_notes' => $request->admin_notes,
            'reviewed_at' => now(),
            'reviewed_by' => Auth::id()
        ]);
        
        return redirect()->route('admin.verification.index')
            ->with('success', 'Verification request has been rejected successfully.');
    }
    
    /**
     * Download verification document.
     *
     * @param  int  $id
     * @param  string  $type
     * @return \Illuminate\Http\Response
     */
    public function downloadDocument($id, $type)
    {
        $verificationRequest = VerificationRequest::findOrFail($id);
        
        if ($type === 'photo') {
            $path = $verificationRequest->photo;
        } elseif ($type === 'document') {
            $path = $verificationRequest->document;
        } else {
            abort(404);
        }
        
        if (!Storage::disk('public')->exists($path)) {
            abort(404);
        }
        
        return response()->download(Storage::disk('public')->path($path));
    }
}