<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\HelpCenterArticle;
use App\Models\HelpCenterTopic;
use Illuminate\Http\Request;

class HelpCenterController extends Controller
{
    public function index(Request $request)
    {
        $searchQuery = trim((string) $request->input('q'));

        $topics = HelpCenterTopic::query()
            ->active()
            ->with(['articles' => function ($query) {
                $query->published()
                    ->orderBy('display_order')
                    ->orderBy('title');
            }])
            ->orderBy('display_order')
            ->orderBy('title')
            ->get();

        $featuredArticles = HelpCenterArticle::published()
            ->with('topic')
            ->where('is_featured', true)
            ->orderBy('display_order')
            ->orderByDesc('updated_at')
            ->take(6)
            ->get();

        $recentArticles = HelpCenterArticle::published()
            ->with('topic')
            ->latest()
            ->take(6)
            ->get();

        $searchResults = collect();
        if ($searchQuery !== '') {
            $searchResults = HelpCenterArticle::published()
                ->with('topic')
                ->where(function ($query) use ($searchQuery) {
                    $query->where('title', 'like', '%' . $searchQuery . '%')
                        ->orWhere('summary', 'like', '%' . $searchQuery . '%')
                        ->orWhere('content', 'like', '%' . $searchQuery . '%');
                })
                ->orderBy('title')
                ->take(20)
                ->get();
        }

        return view('front_end.pages.help_center.index', compact(
            'topics',
            'featuredArticles',
            'recentArticles',
            'searchResults',
            'searchQuery'
        ));
    }

    public function topic(string $slug)
    {
        $topic = HelpCenterTopic::active()
            ->where('slug', $slug)
            ->with(['articles' => function ($query) {
                $query->published()
                    ->orderBy('display_order')
                    ->orderBy('title');
            }])
            ->first();

        if (!$topic) {
            abort(404);
        }

        $relatedTopics = HelpCenterTopic::active()
            ->where('id', '!=', $topic->id)
            ->orderBy('display_order')
            ->orderBy('title')
            ->take(6)
            ->get();

        $featuredArticles = $topic->articles->where('is_featured', true)->take(3);

        return view('front_end.pages.help_center.topic', compact('topic', 'relatedTopics', 'featuredArticles'));
    }

    public function article(string $topicSlug, string $articleSlug)
    {
        $topic = HelpCenterTopic::active()
            ->where('slug', $topicSlug)
            ->first();

        if (!$topic) {
            abort(404);
        }

        $article = HelpCenterArticle::published()
            ->where('slug', $articleSlug)
            ->where('topic_id', $topic->id)
            ->first();

        if (!$article) {
            abort(404);
        }

        $article->increment('view_count');

        $relatedArticles = HelpCenterArticle::published()
            ->where('topic_id', $topic->id)
            ->where('id', '!=', $article->id)
            ->orderBy('display_order')
            ->orderBy('title')
            ->take(5)
            ->get();

        $breadcrumbs = [
            ['label' => 'Help Center', 'url' => route('help-center.index')],
            ['label' => $topic->title, 'url' => route('help-center.topic', $topic->slug)],
            ['label' => $article->title, 'url' => null],
        ];

        return view('front_end.pages.help_center.article', compact('topic', 'article', 'relatedArticles', 'breadcrumbs'));
    }
}
