<?php

namespace App\Http\Requests\Auth;

use Illuminate\Auth\Events\Lockout;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class LoginRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\Rule|array|string>
     */
    public function rules(): array
    {
        return [
            'email' => ['required', 'string', 'email'],
            'password' => ['required', 'string'],
        ];
    }

    /**
     * Attempt to authenticate the request's credentials.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function authenticate(): void
    {
        if ($this->loginAttemptsEnabled()) {
            $this->ensureIsNotRateLimited();
        }

        $credentials = $this->only('email', 'password');
        $credentials['status'] = 1;

        if (! Auth::attempt($credentials, $this->boolean('remember'))) {
            if ($this->loginAttemptsEnabled()) {
                RateLimiter::hit($this->throttleKey(), $this->loginLockSeconds());
            }

            throw ValidationException::withMessages([
                'email' => trans('auth.failed'),
            ]);
        }

        if ($this->loginAttemptsEnabled()) {
            RateLimiter::clear($this->throttleKey());
        }
    }

    /**
     * Ensure the login request is not rate limited.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function ensureIsNotRateLimited(): void
    {
        if (! $this->loginAttemptsEnabled()) {
            return;
        }

        if (! RateLimiter::tooManyAttempts($this->throttleKey(), $this->maxLoginAttempts())) {
            return;
        }

        event(new Lockout($this));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        throw ValidationException::withMessages([
            'email' => trans('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the rate limiting throttle key for the request.
     */
    public function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->string('email')).'|'.$this->ip());
    }

    /**
     * Determine if login attempt limiting is enabled via settings.
     */
    public function loginAttemptsEnabled(): bool
    {
        return (int) setting('prevent_login_attempts', 1) === 1;
    }

    /**
     * Retrieve the maximum number of login attempts allowed before lockout.
     */
    public function maxLoginAttempts(): int
    {
        $attempts = (int) setting('login_attempts', 5);

        return max($attempts, 1);
    }

    /**
     * Retrieve the login lockout duration in seconds.
     */
    public function loginLockSeconds(): int
    {
        $minutes = (int) setting('login_locktime', 30);

        return max($minutes, 1) * 60;
    }
}
