<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Blacklist extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'blacklist';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'value',
        'type',
        'notes',
    ];

    /**
     * Check if a value is blacklisted
     *
     * @param string $value
     * @param string $type
     * @return bool
     */
    public static function isBlacklisted($value, $type)
    {
        return self::where('value', $value)
            ->where('type', $type)
            ->exists();
    }

    /**
     * Check if an email is blacklisted (including domain check)
     *
     * @param string $email
     * @return bool
     */
    public static function isEmailBlacklisted($email)
    {
        // Check if the exact email is blacklisted
        $exactMatch = self::where('value', $email)
            ->where('type', 'email')
            ->exists();
            
        if ($exactMatch) {
            return true;
        }
        
        // Check if the email domain is blacklisted
        $parts = explode('@', $email);
        if (count($parts) === 2) {
            $domain = $parts[1];
            return self::where('value', $domain)
                ->where('type', 'email_domain')
                ->exists();
        }
        
        return false;
    }

    /**
     * Check if an IP is blacklisted
     *
     * @param string $ip
     * @return bool
     */
    public static function isIpBlacklisted($ip)
    {
        return self::where('value', $ip)
            ->where('type', 'ip')
            ->exists();
    }

    /**
     * Check if a username is blacklisted
     *
     * @param string $username
     * @return bool
     */
    public static function isUsernameBlacklisted($username)
    {
        return self::where('value', $username)
            ->where('type', 'username')
            ->exists();
    }
} 