<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Friend extends Model
{
    use HasFactory;

    protected $fillable = [
        'friend_one',
        'friend_two',
        'status',
        'accepted_time',
        'role',
    ];



    public function friendOne()
    {
        return $this->belongsTo(User::class, 'friend_one');
    }

    public function friendTwo()
    {
        return $this->belongsTo(User::class, 'friend_two');
    }

    /**
     * Get the count of mutual friends between two users.
     *
     * @param int $userId
     * @param int $otherUserId
     * @return int
     */
    public static function getMutualFriendsCount($userId, $otherUserId)
    {
        $userFriends = Friend::where('friend_one', $userId)
            ->orWhere('friend_two', $userId)
            ->pluck('friend_one', 'friend_two')
            ->flatten()
            ->unique();

        $otherUserFriends = Friend::where('friend_one', $otherUserId)
            ->orWhere('friend_two', $otherUserId)
            ->pluck('friend_one', 'friend_two')
            ->flatten()
            ->unique();

        $mutualFriends = $userFriends->intersect($otherUserFriends);

        return $mutualFriends->count();
    }

    /**
     * Determine the pending friend request status between two users.
     *
     * Returns:
     * - 'sent'     when $userId has sent a request to $otherUserId that is still pending.
     * - 'received' when $otherUserId has sent a request to $userId that is still pending.
     * - null       when there is no pending request between the users.
     *
     * @param int $userId
     * @param int $otherUserId
     * @return string|null
     */
    public static function pendingRequestStatus($userId, $otherUserId)
    {
        $sent = self::where('friend_one', $userId)
            ->where('friend_two', $otherUserId)
            ->where('status', 0)
            ->exists();

        if ($sent) {
            return 'sent';
        }

        $received = self::where('friend_one', $otherUserId)
            ->where('friend_two', $userId)
            ->where('status', 0)
            ->exists();

        if ($received) {
            return 'received';
        }

        return null;
    }


    /**
     * Check if two users are friends.
     *
     * @param int $userId
     * @param int $friendId
     * @return bool
     */
    public static function isFriend($userId, $friendId)
    {
        return self::where(function ($query) use ($userId, $friendId) {
            $query->where(function ($q) use ($userId, $friendId) {
                $q->where('friend_one', $userId)
                    ->where('friend_two', $friendId);
            })
                ->orWhere(function ($q) use ($userId, $friendId) {
                    $q->where('friend_one', $friendId)
                        ->where('friend_two', $userId);
                });
        })
            ->where('status', 1)
            ->exists();
    }

    /**
     * Get a list of friends for a given user ID.
     *
     * @param int $userId
     * @return \Illuminate\Support\Collection
     */
    public static function getFriends($userId)
    {
        return self::where(function ($query) use ($userId) {
            $query->where('friend_one', $userId)
                ->orWhere('friend_two', $userId);
        })
            ->where('status', 1) // Assuming 'accepted' indicates a confirmed friendship
            ->get();
    }

    public static function getUserFriends($userId)
    {
        // This is where the actual query logic resides. The controller just calls this method.
        return self::with(['friendOne', 'friendTwo'])
            ->where(function ($query) use ($userId) {
                $query->where('friend_one', $userId)
                    ->orWhere('friend_two', $userId);
            })
            ->where('status', 1) // Assuming 'accepted' indicates confirmed friendship
            ->get();
    }
}
