<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Wallet extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'balance',
        'pending_balance',
        'currency',
        'is_active'
    ];

    /**
     * Get the user that owns the wallet.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the transactions for the wallet.
     */
    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    /**
     * Add funds to the wallet.
     *
     * @param float $amount
     * @param string $description
     * @param string $referenceType
     * @param int|null $referenceId
     * @param array|null $metadata
     * @return Transaction
     */
    public function addFunds($amount, $description, $referenceType = null, $referenceId = null, $metadata = null)
    {
        $this->balance += $amount;
        $this->save();

        return Transaction::create([
            'user_id' => $this->user_id,
            'wallet_id' => $this->id,
            'type' => 'credit',
            'amount' => $amount,
            'description' => $description,
            'reference_type' => $referenceType,
            'reference_id' => $referenceId,
            'metadata' => $metadata ? json_encode($metadata) : null,
            'processed_at' => now(),
        ]);
    }

    /**
     * Deduct funds from the wallet.
     *
     * @param float $amount
     * @param string $description
     * @param string $referenceType
     * @param int|null $referenceId
     * @param array|null $metadata
     * @return Transaction|bool
     */
    public function deductFunds($amount, $description, $referenceType = null, $referenceId = null, $metadata = null)
    {
        if ($this->balance < $amount) {
            return false;
        }

        $this->balance -= $amount;
        $this->save();

        return Transaction::create([
            'user_id' => $this->user_id,
            'wallet_id' => $this->id,
            'type' => 'debit',
            'amount' => $amount,
            'description' => $description,
            'reference_type' => $referenceType,
            'reference_id' => $referenceId,
            'metadata' => $metadata ? json_encode($metadata) : null,
            'processed_at' => now(),
        ]);
    }

    /**
     * Add pending funds to the wallet.
     *
     * @param float $amount
     * @param string $description
     * @param string $referenceType
     * @param int|null $referenceId
     * @param array|null $metadata
     * @return Transaction
     */
    public function addPendingFunds($amount, $description, $referenceType = null, $referenceId = null, $metadata = null)
    {
        $this->pending_balance += $amount;
        $this->save();

        return Transaction::create([
            'user_id' => $this->user_id,
            'wallet_id' => $this->id,
            'type' => 'pending',
            'amount' => $amount,
            'description' => $description,
            'reference_type' => $referenceType,
            'reference_id' => $referenceId,
            'metadata' => $metadata ? json_encode($metadata) : null,
            'processed_at' => null,
        ]);
    }

    /**
     * Convert pending funds to available balance.
     *
     * @param Transaction $transaction
     * @return bool
     */
    public function approvePendingTransaction(Transaction $transaction)
    {
        if ($transaction->type !== 'pending' || $transaction->wallet_id !== $this->id) {
            return false;
        }

        $amount = $transaction->amount;
        
        if ($this->pending_balance < $amount) {
            return false;
        }

        $this->pending_balance -= $amount;
        $this->balance += $amount;
        $this->save();

        $transaction->type = 'completed';
        $transaction->processed_at = now();
        $transaction->save();

        return true;
    }
} 