<?php

namespace App\Policies\Messenger;

use App\Models\User;
use App\Models\Messenger\Conversation;
use App\Models\Messenger\Block;
use Illuminate\Auth\Access\Response;

class ConversationPolicy
{
    public function view(User $user, Conversation $conversation): bool
    {
        return $conversation->isParticipant($user->id);
    }

    public function update(User $user, Conversation $conversation): bool
    {
        if (!$conversation->isParticipant($user->id)) {
            return false;
        }

        // Only group conversations can be updated
        if ($conversation->isDirectMessage()) {
            return false;
        }

        $participant = $conversation->getParticipant($user->id);
        return $participant && $participant->isAdmin();
    }

    public function delete(User $user, Conversation $conversation): bool
    {
        if (!$conversation->isParticipant($user->id)) {
            return false;
        }

        $participant = $conversation->getParticipant($user->id);
        return $participant && $participant->isOwner();
    }

    public function leave(User $user, Conversation $conversation): bool
    {
        if (!$conversation->isParticipant($user->id)) {
            return false;
        }

        // Can't leave if you're the only owner of a group
        if ($conversation->isGroup()) {
            $participant = $conversation->getParticipant($user->id);
            if ($participant && $participant->isOwner()) {
                $otherOwners = $conversation->participants()
                    ->where('user_id', '!=', $user->id)
                    ->where('role', 'owner')
                    ->count();
                
                return $otherOwners > 0;
            }
        }

        return true;
    }

    public function addParticipants(User $user, Conversation $conversation): bool
    {
        if (!$conversation->isParticipant($user->id)) {
            return false;
        }

        // Only group conversations can have participants added
        if ($conversation->isDirectMessage()) {
            return false;
        }

        $participant = $conversation->getParticipant($user->id);
        return $participant && $participant->isAdmin();
    }

    public function removeParticipants(User $user, Conversation $conversation): bool
    {
        if (!$conversation->isParticipant($user->id)) {
            return false;
        }

        $participant = $conversation->getParticipant($user->id);
        return $participant && $participant->isAdmin();
    }

    public function sendMessage(User $user, Conversation $conversation): bool
    {
        return $conversation->isParticipant($user->id);
    }

    public function createDirectMessage(User $user, User $otherUser): bool
    {
        // Can't create DM with yourself
        if ($user->id === $otherUser->id) {
            return false;
        }

        // Check if blocked
        return !Block::isBlocked($user->id, $otherUser->id);
    }

    public function createGroup(User $user): bool
    {
        // Basic rate limiting could be implemented here
        return true;
    }
}
