<?php

namespace Database\Seeders;

use App\Models\HelpCenterArticle;
use App\Models\HelpCenterTopic;
use Illuminate\Database\Seeder;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class HelpCenterSeeder extends Seeder
{
    /**
     * Seed the application's help center content with realistic data.
     */
    public function run(): void
    {
        $topics = [
            [
                'slug' => 'getting-started',
                'title' => 'Getting Started',
                'icon' => 'bi bi-rocket-takeoff',
                'description' => 'Kick off your experience with guided tours, profile best practices, and tips that help new members feel at home from day one.',
                'display_order' => 1,
                'is_active' => true,
                'meta_title' => 'Getting Started Guide',
                'meta_description' => 'Learn how to set up your account, personalize your profile, and make your first connections.',
                'articles' => [
                    [
                        'slug' => 'create-your-account',
                        'title' => 'Create your account in minutes',
                        'summary' => 'Everything you need to know about signing up, verifying your email, and securing your login.',
                        'hero_image' => 'https://images.unsplash.com/photo-1521737604893-d14cc237f11d?w=1200',
                        'content' => <<<HTML
<p>Ready to join the community? Registration is intentionally streamlined so you can start sharing right away.</p>
<h2>Step 1 — Provide your basics</h2>
<p>Use a real email address and pick a unique username. Usernames help people find you, so avoid inside jokes that may age quickly.</p>
<h2>Step 2 — Confirm your email</h2>
<p>We send a short-lived verification link. Confirming earns you access to password recovery, device history, and advanced privacy controls.</p>
<h2>Step 3 — Secure your login</h2>
<p>Set a password that blends length with randomness. Aim for at least 12 characters, and enable two-factor authentication from <strong>Settings → Security</strong> once you are signed in.</p>
HTML,
                        'estimated_reading_time' => 4,
                        'display_order' => 1,
                        'is_featured' => true,
                        'is_published' => true,
                        'meta_title' => 'How to create an account',
                        'meta_description' => 'Follow these three steps to open an account securely and start participating right away.',
                    ],
                    [
                        'slug' => 'complete-your-profile',
                        'title' => 'Complete your profile for instant credibility',
                        'summary' => 'First impressions travel fast. Learn how to polish your profile so people know who they are talking to.',
                        'hero_image' => 'https://images.unsplash.com/photo-1521572163474-6864f9cf17ab?w=1200',
                        'content' => <<<HTML
<p>A thoughtful profile puts faces to posts and makes conversations feel personal. It also helps moderators keep spam away.</p>
<ul>
    <li><strong>Profile photo:</strong> Choose a clear, well-lit image. Group photos and logos work, but simple headshots build trust fastest.</li>
    <li><strong>Headline:</strong> Summarize what you love talking about in one sentence. Think “Product designer obsessed with inclusive tech”.</li>
    <li><strong>Bio:</strong> Share a few highlights such as current projects, passions, or fun facts. Two to three short paragraphs keep attention.</li>
    <li><strong>Interests:</strong> Select topics you care about so recommendations stay relevant.</li>
</ul>
<p>Return to your profile every few months. Small tweaks tell contacts you are active and approachable.</p>
HTML,
                        'estimated_reading_time' => 5,
                        'display_order' => 2,
                        'is_featured' => false,
                        'is_published' => true,
                        'meta_title' => 'Profile completion checklist',
                        'meta_description' => 'High-impact tips for crafting an authentic profile that encourages meaningful connections.',
                    ],
                    [
                        'slug' => 'personalize-your-feed',
                        'title' => 'Personalize your feed in under five minutes',
                        'summary' => 'Dial in recommendations by following people, topics, and spaces that match your interests.',
                        'hero_image' => 'https://images.unsplash.com/photo-1517841905240-472988babdf9?w=1200',
                        'content' => <<<HTML
<p>Your feed is designed to adapt. The more signals you provide, the better it becomes at surfacing conversations you care about.</p>
<ol>
    <li>Follow at least five profiles you admire. Tap the follow button on any profile or suggestion card.</li>
    <li>Join two or three spaces dedicated to your favorite topics. Spaces have specialized discovery carousels.</li>
    <li>Mute topics that drain your energy. Muted subjects will no longer appear in explore modules.</li>
    <li>Use the “Show me more like this” control on posts you enjoy. It trains the ranking model instantly.</li>
</ol>
<p>This small routine is enough to curate a feed that feels personal and current.</p>
HTML,
                        'estimated_reading_time' => 3,
                        'display_order' => 3,
                        'is_featured' => false,
                        'is_published' => true,
                        'meta_title' => 'Tune your feed quickly',
                        'meta_description' => 'Short checklist to customize your home feed and mute noise you do not want to see.',
                    ],
                ],
            ],
            [
                'slug' => 'privacy-and-security',
                'title' => 'Privacy & Security',
                'icon' => 'bi bi-shield-lock',
                'description' => 'Control how your information is shared, keep your account safe, and understand how reporting keeps the community healthy.',
                'display_order' => 2,
                'is_active' => true,
                'meta_title' => 'Privacy and Security Resources',
                'meta_description' => 'Safeguard your account and fine-tune visibility settings with these practical guides.',
                'articles' => [
                    [
                        'slug' => 'control-who-sees-your-content',
                        'title' => 'Control who can see what you share',
                        'summary' => 'Adjust visibility settings per post or at the account level so you always share with the right audience.',
                        'hero_image' => 'https://images.unsplash.com/photo-1525182008055-f88b95ff7980?w=1200',
                        'content' => <<<HTML
<p>Privacy settings are flexible so you can tailor each conversation.</p>
<h2>Account-wide visibility</h2>
<p>Visit <strong>Settings → Privacy</strong> to pick between public, friends-only, or custom lists. Custom lists let you combine colleagues, family members, and close friends.</p>
<h2>Per-post controls</h2>
<p>When drafting a post, tap the visibility pill next to the publish button. You can limit replies, hide the post from search, or restrict it to a single list. Your preference is remembered for the next post.</p>
<p>Remember: you can archive or delete any post later. Archiving hides it from every profile except yours.</p>
HTML,
                        'estimated_reading_time' => 4,
                        'display_order' => 1,
                        'is_featured' => true,
                        'is_published' => true,
                        'meta_title' => 'Share content with confidence',
                        'meta_description' => 'Learn how to adjust visibility per audience and keep sensitive updates private.',
                    ],
                    [
                        'slug' => 'keep-your-account-secure',
                        'title' => 'Keep your account secure on every device',
                        'summary' => 'Enable layered protection, spot unusual sessions, and regain access if you lose a phone.',
                        'hero_image' => 'https://images.unsplash.com/photo-1483478550801-ceba5fe50e8e?w=1200',
                        'content' => <<<HTML
<p>Security works best when it runs quietly in the background. A few proactive steps go a long way.</p>
<ul>
    <li><strong>Two-factor authentication:</strong> Choose between authenticator apps, SMS codes, or hardware keys. Set a backup method in case your primary option is unavailable.</li>
    <li><strong>Login alerts:</strong> Turn on alerts to receive a push notification when a new device signs in.</li>
    <li><strong>Session management:</strong> Review active sessions under <strong>Settings → Security → Devices</strong>. Sign out from locations you do not recognize.</li>
    <li><strong>Recovery code:</strong> Download the one-time recovery code and store it offline. It bypasses second-factor prompts during emergencies.</li>
</ul>
<p>If you suspect suspicious activity, change your password immediately and contact support with the device details shown in the security log.</p>
HTML,
                        'estimated_reading_time' => 6,
                        'display_order' => 2,
                        'is_featured' => false,
                        'is_published' => true,
                        'meta_title' => 'Account security checklist',
                        'meta_description' => 'Layered defenses—from 2FA to session alerts—keep your account secure everywhere you log in.',
                    ],
                    [
                        'slug' => 'report-problems-respectfully',
                        'title' => 'Report problems and keep conversations respectful',
                        'summary' => 'Know when to flag content, how reports are reviewed, and what feedback you will receive.',
                        'hero_image' => 'https://images.unsplash.com/photo-1485217988980-11786ced9454?w=1200',
                        'content' => <<<HTML
<p>Community standards are enforced with a mix of human review and automation. Reports are confidential, and learning how they work empowers everyone.</p>
<h2>How to report</h2>
<p>Click the three-dot menu on any post, comment, or profile and choose <strong>Report</strong>. Select the reason that best matches the issue and provide context if prompted.</p>
<h2>What happens next</h2>
<p>Reports enter a queue where trained reviewers assess them in the order received, prioritizing safety concerns. You will receive a notification summarizing the outcome once the review is complete.</p>
<p>If the issue is urgent or involves immediate danger, contact local authorities first and then alert our safety team with as much detail as possible.</p>
HTML,
                        'estimated_reading_time' => 5,
                        'display_order' => 3,
                        'is_featured' => false,
                        'is_published' => true,
                        'meta_title' => 'Understand the reporting process',
                        'meta_description' => 'Guidance on filing effective reports and what to expect once the safety team reviews them.',
                    ],
                ],
            ],
            [
                'slug' => 'creators-and-monetization',
                'title' => 'Creators & Monetization',
                'icon' => 'bi bi-camera-reels',
                'description' => 'Turn your audience into a community, launch paid offerings, and analyze what is working using built-in creator tools.',
                'display_order' => 3,
                'is_active' => true,
                'meta_title' => 'Creators and Monetization',
                'meta_description' => 'Grow your presence with analytics, paid memberships, and collaboration tools built for creators.',
                'articles' => [
                    [
                        'slug' => 'launch-memberships',
                        'title' => 'Launch memberships that feel personal',
                        'summary' => 'Build a membership program with tiers, exclusive updates, and automatic renewals.',
                        'hero_image' => 'https://images.unsplash.com/photo-1520813792240-56fc4a3765a7?w=1200',
                        'content' => <<<HTML
<p>Memberships create closer circles and recurring income. Before launching, outline what supporters will receive each month.</p>
<ol>
    <li><strong>Design your tiers:</strong> Most creators start with two or three tiers. Offer meaningful perks such as behind-the-scenes posts, live office hours, or early access drops.</li>
    <li><strong>Set pricing intentionally:</strong> Price tiers based on time and value. You can adjust pricing later, but up-front clarity builds trust.</li>
    <li><strong>Enable welcome flows:</strong> Craft a welcome note and automated checklist so new members feel appreciated immediately.</li>
    <li><strong>Track retention:</strong> Use the membership analytics tab to spot churn. Experiment with milestone rewards when loyalty dips.</li>
</ol>
HTML,
                        'estimated_reading_time' => 7,
                        'display_order' => 1,
                        'is_featured' => true,
                        'is_published' => true,
                        'meta_title' => 'Membership launch guide',
                        'meta_description' => 'Plan tiers, perks, and pricing to roll out memberships that excite your supporters.',
                    ],
                    [
                        'slug' => 'collaborate-with-brands',
                        'title' => 'Collaborate with brands without losing your voice',
                        'summary' => 'Work with partners who align with your style and keep sponsored content transparent.',
                        'hero_image' => 'https://images.unsplash.com/photo-1545239351-1141bd82e8a6?w=1200',
                        'content' => <<<HTML
<p>Brand partnerships expand your reach when done thoughtfully.</p>
<p><strong>Find the right fit:</strong> Use the creator marketplace filters to identify brands speaking to the same audience. Review their previous activations to gauge tone.</p>
<p><strong>Co-create briefs:</strong> Outline deliverables, deadlines, and non-negotiables in a shared workspace. Transparent expectations protect both parties.</p>
<p><strong>Label clearly:</strong> Use the “Paid partnership” toggle when publishing. Readers spot authenticity quickly, and transparency keeps trust high.</p>
<p><strong>Measure impact:</strong> Track unique links or discount codes. Share post-campaign recaps that highlight reach, clicks, and sentiment.</p>
HTML,
                        'estimated_reading_time' => 4,
                        'display_order' => 2,
                        'is_featured' => false,
                        'is_published' => true,
                        'meta_title' => 'Brand collaboration checklist',
                        'meta_description' => 'Strategies for partnering with brands while staying authentic and transparent with your audience.',
                    ],
                    [
                        'slug' => 'interpret-your-analytics',
                        'title' => 'Interpret your analytics like a strategist',
                        'summary' => 'Read your dashboard signals to understand what content resonates and when to experiment.',
                        'hero_image' => 'https://images.unsplash.com/photo-1525182008055-f88b95ff7980?w=1200',
                        'content' => <<<HTML
<p>Your analytics home reveals patterns that spreadsheets often miss.</p>
<ul>
    <li><strong>Reach vs. engaged viewers:</strong> A high reach but low engagement means your hook needs work. Rephrase intros or swap thumbnails.</li>
    <li><strong>Retention chart:</strong> Look for drop-off cliffs. Add timestamps and chapter markers at segments where people leave.</li>
    <li><strong>Conversion funnels:</strong> If membership conversions dip, refresh your welcome offer or highlight member-only outcomes in public posts.</li>
    <li><strong>Time-of-day insights:</strong> Use the scheduler to post when your audience is most active. Weekly trends often appear after a month of publishing.</li>
</ul>
<p>Set aside time every Monday to review the dashboard. Treat analytics as feedback rather than pressure.</p>
HTML,
                        'estimated_reading_time' => 5,
                        'display_order' => 3,
                        'is_featured' => false,
                        'is_published' => true,
                        'meta_title' => 'Analytics interpretation guide',
                        'meta_description' => 'Transform dashboard metrics into actionable experiments that grow your community.',
                    ],
                ],
            ],
        ];

        foreach ($topics as $topicData) {
            $articlePayloads = Arr::pull($topicData, 'articles', []);
            $slug = $topicData['slug'] ?? Str::slug($topicData['title']);

            $topic = HelpCenterTopic::updateOrCreate(
                ['slug' => $slug],
                Arr::except($topicData, ['slug'])
            );

            foreach ($articlePayloads as $articleData) {
                $articleSlug = $articleData['slug'] ?? Str::slug($articleData['title']);

                HelpCenterArticle::updateOrCreate(
                    ['slug' => $articleSlug],
                    array_merge(
                        Arr::except($articleData, ['slug']),
                        ['topic_id' => $topic->id]
                    )
                );
            }
        }
    }
}
