@extends('admin.layouts.home')

@section('title', 'Product Categories')

@section('styles')
<style>
    .category-card {
        transition: transform 0.3s ease, box-shadow 0.3s ease;
        border-radius: 10px;
        overflow: hidden;
    }
    
    .category-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
    }
    
    .action-btn {
        width: 32px;
        height: 32px;
        border-radius: 50%;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
    }
    
    .action-btn:hover {
        transform: translateY(-2px);
    }
    
    .category-icon {
        width: 40px;
        height: 40px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
    }
</style>
@endsection

@section('content')
<div class="container-fluid px-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="mt-4">Product Categories</h1>
        <div>
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCategoryModal">
                <i class="fas fa-plus me-1"></i> Add Category
            </button>
            <a href="{{ route('admin.marketplace.dashboard') }}" class="btn btn-outline-primary ms-2">
                <i class="fas fa-chart-line me-1"></i> Dashboard
            </a>
        </div>
    </div>
    
    <!-- Categories List -->
    <div class="row">
        @foreach($categories as $category)
            <div class="col-xl-3 col-md-6 mb-4">
                <div class="card category-card h-100">
                    <div class="card-body">
                        <div class="mb-3">
                            <div class="d-flex align-items-center">
                                <div class="category-icon bg-primary-subtle text-primary me-3">
                                    <i class="fas {{ $category->icon ?? 'fa-tag' }}"></i>
                                </div>
                                <h5 class="mb-0">{{ $category->name }}</h5>
                            </div>
                            <hr>
                            <div>
                                <button type="button" class="btn btn-sm btn-outline-primary action-btn me-1" 
                                        data-bs-toggle="modal" data-bs-target="#editCategoryModal{{ $category->id }}" 
                                        title="Edit Category">
                                    <i class="fas fa-edit"></i> Edit
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-danger action-btn" 
                                        data-bs-toggle="modal" data-bs-target="#deleteCategoryModal{{ $category->id }}" 
                                        title="Delete Category">
                                    <i class="fas fa-trash"></i> Delete
                                </button>
                            </div>
                        </div>
                        
                        <p class="text-muted mb-3">
                            {{ Str::limit($category->description, 100) ?? 'No description available.' }}
                        </p>
                        
                        <div class="d-flex justify-content-between align-items-center">
                            <span class="badge bg-primary rounded-pill">
                                {{ $category->products_count }} Products
                            </span>
                            <a href="{{ route('admin.marketplace.products', ['category' => $category->id]) }}" class="btn btn-sm btn-outline-secondary">
                                View Products
                            </a>
                        </div>
                    </div>
                </div>
                
                <!-- Edit Category Modal -->
                <div class="modal fade" id="editCategoryModal{{ $category->id }}" tabindex="-1" aria-labelledby="editCategoryModalLabel{{ $category->id }}" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="editCategoryModalLabel{{ $category->id }}">Edit Category</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <form action="{{ route('admin.marketplace.categories.update', $category->id) }}" method="POST">
                                @csrf
                                @method('PUT')
                                <div class="modal-body">
                                    <div class="mb-3">
                                        <label for="name{{ $category->id }}" class="form-label">Category Name</label>
                                        <input type="text" class="form-control" id="name{{ $category->id }}" name="name" value="{{ $category->name }}" required>
                                    </div>
                                    <div class="mb-3">
                                        <label for="description{{ $category->id }}" class="form-label">Description</label>
                                        <textarea class="form-control" id="description{{ $category->id }}" name="description" rows="3">{{ $category->description }}</textarea>
                                    </div>
                                    <div class="mb-3">
                                        <label for="icon{{ $category->id }}" class="form-label">Icon (FontAwesome class)</label>
                                        <div class="input-group">
                                            <span class="input-group-text"><i class="fas {{ $category->icon ?? 'fa-tag' }}"></i></span>
                                            <input type="text" class="form-control" id="icon{{ $category->id }}" name="icon" value="{{ $category->icon }}" placeholder="fa-tag">
                                        </div>
                                        <div class="form-text">Enter FontAwesome icon class (e.g., fa-tag, fa-shirt, fa-laptop)</div>
                                    </div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                    <button type="submit" class="btn btn-primary">Update Category</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
                
                <!-- Delete Category Modal -->
                <div class="modal fade" id="deleteCategoryModal{{ $category->id }}" tabindex="-1" aria-labelledby="deleteCategoryModalLabel{{ $category->id }}" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="deleteCategoryModalLabel{{ $category->id }}">Confirm Delete</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <div class="modal-body">
                                <p>Are you sure you want to delete the category <strong>{{ $category->name }}</strong>?</p>
                                @if($category->products_count > 0)
                                    <div class="alert alert-warning">
                                        <i class="fas fa-exclamation-triangle me-1"></i>
                                        This category has {{ $category->products_count }} products associated with it. Deleting this category may affect these products.
                                    </div>
                                @endif
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <form action="{{ route('admin.marketplace.categories.destroy', $category->id) }}" method="POST">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="btn btn-danger">Delete Category</button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        @endforeach
        
        <!-- Add Category Card -->
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card category-card h-100 border-dashed">
                <div class="card-body d-flex flex-column align-items-center justify-content-center" style="min-height: 200px;">
                    <div class="text-center mb-3">
                        <div class="category-icon bg-light text-secondary mb-3 mx-auto">
                            <i class="fas fa-plus"></i>
                        </div>
                        <h5>Add New Category</h5>
                        <p class="text-muted">Create a new product category</p>
                    </div>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCategoryModal">
                        <i class="fas fa-plus me-1"></i> Add Category
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Category Modal -->
    <div class="modal fade" id="addCategoryModal" tabindex="-1" aria-labelledby="addCategoryModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addCategoryModalLabel">Add New Category</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="{{ route('admin.marketplace.categories.store') }}" method="POST">
                    @csrf
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="name" class="form-label">Category Name</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                        </div>
                        <div class="mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                        </div>
                        <div class="mb-3">
                            <label for="icon" class="form-label">Icon (FontAwesome class)</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="fas fa-tag"></i></span>
                                <input type="text" class="form-control" id="icon" name="icon" placeholder="fa-tag">
                            </div>
                            <div class="form-text">Enter FontAwesome icon class (e.g., fa-tag, fa-shirt, fa-laptop)</div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Create Category</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Preview icon when typing in the icon field
        const iconInputs = document.querySelectorAll('input[name="icon"]');
        iconInputs.forEach(input => {
            const iconPreview = input.parentElement.querySelector('.input-group-text i');
            
            input.addEventListener('input', function() {
                // Remove existing classes except 'fas'
                iconPreview.className = 'fas';
                
                // Add the new icon class if it's not empty
                if (this.value) {
                    iconPreview.classList.add(this.value);
                } else {
                    iconPreview.classList.add('fa-tag');
                }
            });
        });
    });
</script>
@endsection 