@extends('admin.layouts.home')

@section('title', 'Marketplace Dashboard')

@section('styles')
<style>
    .stat-card {
        border-radius: 10px;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.05);
        transition: transform 0.3s ease;
    }
    
    .stat-card:hover {
        transform: translateY(-5px);
    }
    
    .stat-icon {
        width: 50px;
        height: 50px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
    }
    
    .chart-container {
        position: relative;
        height: 300px;
        margin-bottom: 20px;
    }
    
    .product-image {
        width: 50px;
        height: 50px;
        object-fit: cover;
        border-radius: 8px;
    }
    
    .order-item {
        transition: all 0.3s ease;
    }
    
    .order-item:hover {
        background-color: rgba(0, 0, 0, 0.02);
    }
    
    .status-badge {
        padding: 5px 10px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 500;
    }
</style>
@endsection

@section('content')
<div class="container-fluid px-4">
    <h1 class="mt-4 mb-4">Marketplace Dashboard</h1>
    
    <!-- Stats Row -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6">
            <div class="card stat-card bg-primary text-white mb-4">
                <div class="card-body d-flex align-items-center">
                    <div class="stat-icon bg-white text-primary me-3">
                        <i class="fas fa-box fa-2x"></i>
                    </div>
                    <div>
                        <h5 class="mb-0">{{ $totalProducts }}</h5>
                        <div class="small">Total Products</div>
                    </div>
                </div>
                <div class="card-footer d-flex align-items-center justify-content-between">
                    <a class="small text-white stretched-link" href="{{ route('admin.marketplace.products') }}">View Products</a>
                    <div class="small text-white"><i class="fas fa-angle-right"></i></div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6">
            <div class="card stat-card bg-success text-white mb-4">
                <div class="card-body d-flex align-items-center">
                    <div class="stat-icon bg-white text-success me-3">
                        <i class="fas fa-store fa-2x"></i>
                    </div>
                    <div>
                        <h5 class="mb-0">{{ $totalStores }}</h5>
                        <div class="small">Total Stores</div>
                    </div>
                </div>
                <div class="card-footer d-flex align-items-center justify-content-between">
                    <a class="small text-white stretched-link" href="{{ route('admin.marketplace.stores') }}">View Stores</a>
                    <div class="small text-white"><i class="fas fa-angle-right"></i></div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6">
            <div class="card stat-card bg-warning text-white mb-4">
                <div class="card-body d-flex align-items-center">
                    <div class="stat-icon bg-white text-warning me-3">
                        <i class="fas fa-shopping-cart fa-2x"></i>
                    </div>
                    <div>
                        <h5 class="mb-0">{{ $totalOrders }}</h5>
                        <div class="small">Total Orders</div>
                    </div>
                </div>
                <div class="card-footer d-flex align-items-center justify-content-between">
                    <a class="small text-white stretched-link" href="{{ route('admin.marketplace.orders') }}">View Orders</a>
                    <div class="small text-white"><i class="fas fa-angle-right"></i></div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6">
            <div class="card stat-card bg-danger text-white mb-4">
                <div class="card-body d-flex align-items-center">
                    <div class="stat-icon bg-white text-danger me-3">
                        <i class="fas fa-dollar-sign fa-2x"></i>
                    </div>
                    <div>
                        <h5 class="mb-0">${{ number_format($totalRevenue, 2) }}</h5>
                        <div class="small">Total Revenue</div>
                    </div>
                </div>
                <div class="card-footer d-flex align-items-center justify-content-between">
                    <a class="small text-white stretched-link" href="{{ route('admin.marketplace.orders') }}">View Details</a>
                    <div class="small text-white"><i class="fas fa-angle-right"></i></div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Charts Row -->
    <div class="row mb-4">
        <div class="col-xl-8">
            <div class="card stat-card mb-4">
                <div class="card-header">
                    <i class="fas fa-chart-bar me-1"></i>
                    Monthly Sales & Orders
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="monthlySalesChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-4">
            <div class="card stat-card mb-4">
                <div class="card-header">
                    <i class="fas fa-chart-pie me-1"></i>
                    Top Product Categories
                </div>
                <div class="card-body">
                    <div class="chart-container">
                        <canvas id="categoriesChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Recent Orders & Top Products Row -->
    <div class="row">
        <div class="col-xl-6">
            <div class="card stat-card mb-4">
                <div class="card-header">
                    <i class="fas fa-shopping-cart me-1"></i>
                    Recent Orders
                </div>
                <div class="card-body">
                    @if($recentOrders->isEmpty())
                        <div class="alert alert-info">No recent orders found.</div>
                    @else
                        <div class="table-responsive">
                            <table class="table table-borderless">
                                <thead>
                                    <tr>
                                        <th>Order #</th>
                                        <th>Customer</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($recentOrders as $order)
                                        <tr class="order-item">
                                            <td>
                                                <a href="{{ route('admin.marketplace.orders.show', $order->id) }}">
                                                    #{{ $order->order_number }}
                                                </a>
                                            </td>
                                            <td>{{ $order->user->name }}</td>
                                            <td>${{ number_format($order->total_amount, 2) }}</td>
                                            <td>
                                                @php
                                                    $statusClass = [
                                                        'pending' => 'bg-warning',
                                                        'processing' => 'bg-info',
                                                        'completed' => 'bg-success',
                                                        'cancelled' => 'bg-danger',
                                                    ][$order->status] ?? 'bg-secondary';
                                                @endphp
                                                <span class="status-badge {{ $statusClass }} text-white">
                                                    {{ ucfirst($order->status) }}
                                                </span>
                                            </td>
                                            <td>{{ $order->created_at->format('M d, Y') }}</td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        <div class="text-end mt-3">
                            <a href="{{ route('admin.marketplace.orders') }}" class="btn btn-sm btn-primary">View All Orders</a>
                        </div>
                    @endif
                </div>
            </div>
        </div>
        
        <div class="col-xl-6">
            <div class="card stat-card mb-4">
                <div class="card-header">
                    <i class="fas fa-star me-1"></i>
                    Top Selling Products
                </div>
                <div class="card-body">
                    @if($topSellingProducts->isEmpty())
                        <div class="alert alert-info">No products found.</div>
                    @else
                        <div class="table-responsive">
                            <table class="table table-borderless">
                                <thead>
                                    <tr>
                                        <th>Product</th>
                                        <th>Store</th>
                                        <th>Price</th>
                                        <th>Orders</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($topSellingProducts as $product)
                                        <tr class="order-item">
                                            <td class="d-flex align-items-center">
                                                @if($product->thumbnail)
                                                    <img src="{{ asset('storage/' . $product->thumbnail) }}" alt="{{ $product->name }}" class="product-image me-2">
                                                @else
                                                    <div class="product-image bg-light d-flex align-items-center justify-content-center me-2">
                                                        <i class="fas fa-box text-secondary"></i>
                                                    </div>
                                                @endif
                                                <a href="{{ route('admin.marketplace.products.show', $product->id) }}">
                                                    {{ Str::limit($product->name, 30) }}
                                                </a>
                                            </td>
                                            <td>{{ $product->store->name ?? 'N/A' }}</td>
                                            <td>${{ number_format($product->price, 2) }}</td>
                                            <td>{{ $product->orders_count }}</td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        <div class="text-end mt-3">
                            <a href="{{ route('admin.marketplace.products') }}" class="btn btn-sm btn-primary">View All Products</a>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Monthly Sales & Orders Chart
        const monthlySalesCtx = document.getElementById('monthlySalesChart').getContext('2d');
        const monthlySalesChart = new Chart(monthlySalesCtx, {
            type: 'bar',
            data: {
                labels: {!! json_encode(array_keys($monthlySalesData)) !!},
                datasets: [
                    {
                        label: 'Sales ($)',
                        data: {!! json_encode(array_values($monthlySalesData)) !!},
                        backgroundColor: 'rgba(54, 162, 235, 0.5)',
                        borderColor: 'rgba(54, 162, 235, 1)',
                        borderWidth: 1,
                        yAxisID: 'y'
                    },
                    {
                        label: 'Orders',
                        data: {!! json_encode(array_values($monthlyOrdersData)) !!},
                        backgroundColor: 'rgba(255, 99, 132, 0.5)',
                        borderColor: 'rgba(255, 99, 132, 1)',
                        borderWidth: 1,
                        type: 'line',
                        yAxisID: 'y1'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: {
                            display: true,
                            text: 'Sales ($)'
                        }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        grid: {
                            drawOnChartArea: false
                        },
                        title: {
                            display: true,
                            text: 'Orders'
                        }
                    }
                }
            }
        });
        
        // Categories Chart (placeholder - would need actual category data)
        const categoriesCtx = document.getElementById('categoriesChart').getContext('2d');
        const categoriesChart = new Chart(categoriesCtx, {
            type: 'pie',
            data: {
                labels: ['Electronics', 'Clothing', 'Home & Garden', 'Books', 'Others'],
                datasets: [{
                    data: [30, 25, 20, 15, 10],
                    backgroundColor: [
                        'rgba(54, 162, 235, 0.7)',
                        'rgba(255, 99, 132, 0.7)',
                        'rgba(255, 206, 86, 0.7)',
                        'rgba(75, 192, 192, 0.7)',
                        'rgba(153, 102, 255, 0.7)'
                    ],
                    borderColor: [
                        'rgba(54, 162, 235, 1)',
                        'rgba(255, 99, 132, 1)',
                        'rgba(255, 206, 86, 1)',
                        'rgba(75, 192, 192, 1)',
                        'rgba(153, 102, 255, 1)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right'
                    }
                }
            }
        });
    });
</script>
@endsection 