@extends('front_end.layouts.home')

@section('title', 'Messenger - SocialLink')

@php
    $showLeftSidebar = false;
    $showRightSidebar = false;
@endphp

@push('styles')
<style>
/* Modern Messenger Styling */
.messenger-container {
    height: calc(100vh - 120px);
    display: flex;
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    overflow: hidden;
}

.messenger-sidebar {
    width: 350px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    flex-direction: column;
    color: white;
}

.messenger-header {
    padding: 20px;
    background: rgba(255,255,255,0.1);
    backdrop-filter: blur(10px);
}

.messenger-header h2 {
    margin: 0;
    font-size: 1.5rem;
    font-weight: 600;
}

.search-container {
    position: relative;
    margin-top: 15px;
}

.search-input {
    width: 100%;
    padding: 12px 45px 12px 15px;
    border: none;
    border-radius: 25px;
    background: rgba(255,255,255,0.2);
    color: white;
    font-size: 14px;
    backdrop-filter: blur(10px);
}

.search-input::placeholder {
    color: rgba(255,255,255,0.7);
}

.search-icon {
    position: absolute;
    right: 15px;
    top: 50%;
    transform: translateY(-50%);
    color: rgba(255,255,255,0.7);
}

.conversation-list {
    flex: 1;
    overflow-y: auto;
    padding: 0;
}

.conversation-item {
    display: flex;
    align-items: center;
    padding: 15px 20px;
    cursor: pointer;
    transition: all 0.3s ease;
    border-bottom: 1px solid rgba(255,255,255,0.1);
    position: relative;
}

.conversation-item:hover,
.conversation-item.active {
    background: rgba(255,255,255,0.2);
    backdrop-filter: blur(10px);
}

.conversation-avatar {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    margin-right: 15px;
    position: relative;
    background: #fff;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    color: #667eea;
}

.online-indicator {
    position: absolute;
    bottom: 2px;
    right: 2px;
    width: 12px;
    height: 12px;
    background: #00ff88;
    border: 2px solid white;
    border-radius: 50%;
}

.conversation-info {
    flex: 1;
    min-width: 0;
}

.conversation-title {
    font-weight: 600;
    margin: 0 0 5px 0;
    font-size: 16px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.conversation-preview {
    font-size: 14px;
    opacity: 0.8;
    margin: 0;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.conversation-meta {
    text-align: right;
    min-width: 60px;
}

.conversation-time {
    font-size: 12px;
    opacity: 0.7;
    margin-bottom: 5px;
}

.unread-badge {
    background: #ff4757;
    color: white;
    border-radius: 12px;
    padding: 2px 8px;
    font-size: 12px;
    font-weight: 600;
    min-width: 20px;
    text-align: center;
}

.messenger-main {
    flex: 1;
    display: flex;
    flex-direction: column;
    background: #f8f9fa;
}

.chat-header {
    padding: 20px;
    background: white;
    border-bottom: 1px solid #e9ecef;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.chat-user-info {
    display: flex;
    align-items: center;
}

.chat-user-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 15px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: bold;
}

.chat-user-details h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 600;
    color: #2c3e50;
}

.chat-user-status {
    font-size: 14px;
    color: #7f8c8d;
    margin: 0;
}

.chat-actions {
    display: flex;
    gap: 10px;
}

.action-btn {
    width: 40px;
    height: 40px;
    border: none;
    border-radius: 50%;
    background: #f8f9fa;
    color: #6c757d;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
}

.action-btn:hover {
    background: #e9ecef;
    color: #495057;
}

.messages-container {
    flex: 1;
    overflow-y: auto;
    padding: 20px;
    background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
}

.message-group {
    margin-bottom: 20px;
}

.message {
    max-width: 70%;
    margin-bottom: 10px;
    display: flex;
    align-items: flex-end;
    gap: 10px;
}

.message.own {
    margin-left: auto;
    flex-direction: row-reverse;
}

.message-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 12px;
    font-weight: bold;
    flex-shrink: 0;
}

.message-content {
    background: white;
    padding: 12px 16px;
    border-radius: 18px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    position: relative;
    max-width: 100%;
    word-wrap: break-word;
}

.message.own .message-content {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.message-text {
    margin: 0;
    line-height: 1.4;
}

.message-time {
    font-size: 11px;
    opacity: 0.7;
    margin-top: 5px;
    text-align: right;
}

.message.own .message-time {
    text-align: left;
}

.typing-indicator {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 15px 20px;
    background: rgba(255,255,255,0.8);
    border-radius: 18px;
    margin: 10px 0;
    max-width: 200px;
}

.typing-dots {
    display: flex;
    gap: 3px;
}

.typing-dot {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    background: #667eea;
    animation: typing 1.4s infinite ease-in-out;
}

.typing-dot:nth-child(1) { animation-delay: -0.32s; }
.typing-dot:nth-child(2) { animation-delay: -0.16s; }

@keyframes typing {
    0%, 80%, 100% { transform: scale(0); }
    40% { transform: scale(1); }
}

.message-composer {
    padding: 20px;
    background: white;
    border-top: 1px solid #e9ecef;
}

.composer-container {
    display: flex;
    align-items: flex-end;
    gap: 15px;
    background: #f8f9fa;
    border-radius: 25px;
    padding: 10px 15px;
    min-height: 50px;
}

.composer-input {
    flex: 1;
    border: none;
    background: none;
    resize: none;
    font-size: 16px;
    line-height: 1.4;
    max-height: 120px;
    min-height: 20px;
    outline: none;
    padding: 5px 0;
}

.composer-actions {
    display: flex;
    gap: 10px;
    align-items: center;
}

.composer-btn {
    width: 35px;
    height: 35px;
    border: none;
    border-radius: 50%;
    background: none;
    color: #6c757d;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
}

.composer-btn:hover {
    background: #e9ecef;
    color: #495057;
}

.send-btn {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.send-btn:hover {
    transform: scale(1.1);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
}

.emoji-picker {
    position: absolute;
    bottom: 70px;
    right: 20px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 8px 30px rgba(0,0,0,0.15);
    padding: 15px;
    display: none;
    z-index: 1000;
    max-width: 300px;
}

.emoji-picker.show {
    display: block;
}

.emoji-grid {
    display: grid;
    grid-template-columns: repeat(8, 1fr);
    gap: 8px;
    max-height: 200px;
    overflow-y: auto;
}

.emoji-item {
    padding: 8px;
    border: none;
    background: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 20px;
    transition: background-color 0.2s;
}

.emoji-item:hover {
    background: #f8f9fa;
}

.empty-state {
    flex: 1;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    text-align: center;
    color: #6c757d;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 20px;
    opacity: 0.3;
}

/* Responsive Design */
@media (max-width: 768px) {
    .messenger-container {
        height: calc(100vh - 80px);
    }
    
    .messenger-sidebar {
        width: 100%;
        position: absolute;
        z-index: 1000;
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }
    
    .messenger-sidebar.show {
        transform: translateX(0);
    }
    
    .message {
        max-width: 85%;
    }
    
    .composer-container {
        margin: 0 -5px;
    }
}

/* Loading States */
.loading {
    opacity: 0.6;
    pointer-events: none;
}

.loading::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 20px;
    height: 20px;
    margin: -10px 0 0 -10px;
    border: 2px solid #f3f3f3;
    border-top: 2px solid #667eea;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Scrollbar Styling */
.conversation-list::-webkit-scrollbar,
.messages-container::-webkit-scrollbar {
    width: 6px;
}

.conversation-list::-webkit-scrollbar-track,
.messages-container::-webkit-scrollbar-track {
    background: rgba(255,255,255,0.1);
}

.conversation-list::-webkit-scrollbar-thumb,
.messages-container::-webkit-scrollbar-thumb {
    background: rgba(255,255,255,0.3);
    border-radius: 3px;
}

.conversation-list::-webkit-scrollbar-thumb:hover,
.messages-container::-webkit-scrollbar-thumb:hover {
    background: rgba(255,255,255,0.5);
}
</style>
@endpush

@section('content')
<div id="messenger-app" class="messenger-container">
    <!-- Sidebar -->
    <div class="messenger-sidebar" id="messenger-sidebar">
        <div class="messenger-header">
            <h2><i class="fas fa-comments"></i> Messages</h2>
            <div class="search-container">
                <input type="text" 
                       class="search-input" 
                       id="conversation-search" 
                       placeholder="Search conversations..."
                       autocomplete="off">
                <i class="fas fa-search search-icon"></i>
            </div>
        </div>
        
        <div class="conversation-list" id="conversation-list">
            <!-- Conversations will be loaded here -->
        </div>
        
        <div class="sidebar-footer" style="padding: 15px; text-align: center; opacity: 0.7;">
            <small>Real-time messaging powered by SocialLink</small>
        </div>
    </div>

    <!-- Main Chat Area -->
    <div class="messenger-main" id="messenger-main">
        <!-- Chat Header -->
        <div class="chat-header" id="chat-header" style="display: none;">
            <div class="chat-user-info">
                <div class="chat-user-avatar" id="chat-avatar">JD</div>
                <div class="chat-user-details">
                    <h3 id="chat-user-name">Select a conversation</h3>
                    <p class="chat-user-status" id="chat-user-status">to start messaging</p>
                </div>
            </div>
            <div class="chat-actions">
                <button class="action-btn" title="Voice Call" onclick="messenger.startVoiceCall()">
                    <i class="fas fa-phone"></i>
                </button>
                <button class="action-btn" title="Video Call" onclick="messenger.startVideoCall()">
                    <i class="fas fa-video"></i>
                </button>
                <button class="action-btn" title="More Options" onclick="messenger.showOptions()">
                    <i class="fas fa-ellipsis-v"></i>
                </button>
            </div>
        </div>

        <!-- Messages Area -->
        <div class="messages-container" id="messages-container">
            <div class="empty-state" id="empty-state">
                <div class="empty-icon">
                    <i class="fas fa-comments"></i>
                </div>
                <h3>Welcome to SocialLink Messenger</h3>
                <p>Select a conversation from the sidebar to start chatting with your friends and groups.</p>
                <button class="btn btn-primary" onclick="messenger.newConversation()" style="margin-top: 20px;">
                    <i class="fas fa-plus"></i> Start New Conversation
                </button>
            </div>
        </div>

        <!-- Message Composer -->
        <div class="message-composer" id="message-composer" style="display: none;">
            <div class="composer-container">
                <div class="composer-actions">
                    <button class="composer-btn" title="Attach File" onclick="messenger.attachFile()">
                        <i class="fas fa-paperclip"></i>
                    </button>
                    <button class="composer-btn" title="Add Photo" onclick="messenger.attachPhoto()">
                        <i class="fas fa-image"></i>
                    </button>
                    <button class="composer-btn" title="Add Emoji" onclick="messenger.toggleEmoji()">
                        <i class="fas fa-smile"></i>
                    </button>
                </div>
                
                <textarea class="composer-input" 
                          id="message-input" 
                          placeholder="Type a message..."
                          rows="1"></textarea>
                
                <button class="composer-btn send-btn" id="send-btn" onclick="messenger.sendMessage()">
                    <i class="fas fa-paper-plane"></i>
                </button>
            </div>
            
            <!-- Emoji Picker -->
            <div class="emoji-picker" id="emoji-picker">
                <div class="emoji-grid" id="emoji-grid">
                    <!-- Emojis will be loaded here -->
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Hidden file inputs -->
<input type="file" id="file-input" style="display: none;" multiple accept="*/*">
<input type="file" id="photo-input" style="display: none;" multiple accept="image/*,video/*">
@endsection

@push('scripts')
<script src="{{ asset('js/messenger.js') }}"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize messenger
    @auth
    window.messenger = new MessengerApp({
        apiBase: '/api/messenger',
        currentUserId: {{ auth()->id() }},
        @php
            $currentUser = [
                'id' => auth()->id(),
                'username' => optional(auth()->user())->username ?? 'user',
                'first_name' => optional(auth()->user())->first_name ?? '',
                'last_name' => optional(auth()->user())->last_name ?? '',
                'full_name' => trim((optional(auth()->user())->first_name ?? '') . ' ' . (optional(auth()->user())->last_name ?? '')),
                'avatar' => optional(auth()->user())->avatar ?? '/images/default-avatar.png',
            ];
        @endphp

        currentUser: @json($currentUser)
    });
    @else
    window.messenger = new MessengerApp({
        apiBase: '/api/messenger',
        currentUserId: 1,
        currentUser: {
            id: 1,
            username: 'demo_user',
            first_name: 'Demo',
            last_name: 'User',
            full_name: 'Demo User',
            avatar: '/images/default-avatar.png'
        }
    });
    @endauth
    
    // Auto-resize textarea
    const messageInput = document.getElementById('message-input');
    messageInput.addEventListener('input', function() {
        this.style.height = 'auto';
        this.style.height = Math.min(this.scrollHeight, 120) + 'px';
    });
    
    // Send message on Enter (but not Shift+Enter)
    messageInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            messenger.sendMessage();
        }
    });
    
    // File upload handlers
    document.getElementById('file-input').addEventListener('change', function(e) {
        messenger.handleFileUpload(e.target.files);
    });
    
    document.getElementById('photo-input').addEventListener('change', function(e) {
        messenger.handlePhotoUpload(e.target.files);
    });
    
    // Close emoji picker when clicking outside
    document.addEventListener('click', function(e) {
        const emojiPicker = document.getElementById('emoji-picker');
        const emojiBtn = e.target.closest('[onclick="messenger.toggleEmoji()"]');
        
        if (!emojiBtn && !emojiPicker.contains(e.target)) {
            emojiPicker.classList.remove('show');
        }
    });
    
    // Responsive sidebar toggle
    const sidebarToggle = document.createElement('button');
    sidebarToggle.className = 'btn btn-link d-md-none';
    sidebarToggle.innerHTML = '<i class="fas fa-bars"></i>';
    sidebarToggle.style.cssText = 'position: fixed; top: 20px; left: 20px; z-index: 1001;';
    sidebarToggle.onclick = function() {
        document.getElementById('messenger-sidebar').classList.toggle('show');
    };
    document.body.appendChild(sidebarToggle);
});
</script>
@endpush