<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Event;
use App\Models\Post;
use App\Models\EventParticipant;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

class EventPostTest extends TestCase
{
    use RefreshDatabase;

    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create a test user
        $this->user = User::factory()->create();
        
        // Fake storage for file uploads
        Storage::fake('public');
    }

    /** @test */
    public function it_creates_a_post_when_an_event_is_created()
    {
        $this->actingAs($this->user);

        $eventData = [
            'title' => 'Test Event',
            'description' => 'This is a test event',
            'location' => 'Test Location',
            'start_datetime' => '2024-12-25 10:00:00',
            'end_datetime' => '2024-12-25 12:00:00',
            'privacy' => 'public',
            'category' => 'social',
            'allow_guests' => true,
            'require_approval' => false,
        ];

        $response = $this->postJson('/api/event/add', $eventData);

        $response->assertStatus(201);

        // Check that the event was created
        $this->assertDatabaseHas('events', [
            'name' => 'Test Event',
            'user_id' => $this->user->id,
        ]);

        // Check that a post was created for the event
        $event = Event::where('name', 'Test Event')->first();
        $this->assertDatabaseHas('posts', [
            'post_type' => 'event',
            'user_id' => $this->user->id,
            'event_id' => $event->id,
            'post_category' => 'event',
        ]);

        // Check that the post text contains event information
        $post = Post::where('event_id', $event->id)->first();
        $this->assertStringContainsString('Test Event', $post->post_text);
        $this->assertStringContainsString('Test Location', $post->post_text);
    }

    /** @test */
    public function it_creates_a_post_with_cover_image_when_provided()
    {
        $this->actingAs($this->user);

        $file = UploadedFile::fake()->image('event-cover.jpg');

        $eventData = [
            'title' => 'Test Event with Image',
            'description' => 'This is a test event with cover image',
            'location' => 'Test Location',
            'start_datetime' => '2024-12-25 10:00:00',
            'end_datetime' => '2024-12-25 12:00:00',
            'privacy' => 'public',
            'category' => 'social',
            'cover_image' => $file,
        ];

        $response = $this->postJson('/api/event/add', $eventData);

        $response->assertStatus(201);

        // Check that the event was created with cover image
        $event = Event::where('name', 'Test Event with Image')->first();
        $this->assertNotNull($event->cover_image);

        // Check that the post has media
        $post = Post::where('event_id', $event->id)->first();
        $this->assertEquals(1, $post->media_type); // 1 for image
    }

    /** @test */
    public function it_updates_the_post_when_event_is_updated()
    {
        $this->actingAs($this->user);

        // Create an event first
        $event = Event::factory()->create([
            'user_id' => $this->user->id,
            'name' => 'Original Event Name',
            'location' => 'Original Location',
        ]);

        // Create the associated post
        $post = Post::create([
            'post_type' => 'event',
            'user_id' => $this->user->id,
            'event_id' => $event->id,
            'post_text' => 'Original post text',
            'post_category' => 'event',
            'status' => 1,
        ]);

        // Update the event
        $updateData = [
            'event_id' => $event->id,
            'title' => 'Updated Event Name',
            'location' => 'Updated Location',
        ];

        $response = $this->postJson('/api/event/update', $updateData);

        $response->assertStatus(200);

        // Check that the post was updated
        $post->refresh();
        $this->assertStringContainsString('Updated Event Name', $post->post_text);
        $this->assertEquals('Updated Location', $post->location);
    }

    /** @test */
    public function it_deletes_the_post_when_event_is_deleted()
    {
        $this->actingAs($this->user);

        // Create an event
        $event = Event::factory()->create([
            'user_id' => $this->user->id,
        ]);

        // Create the associated post
        $post = Post::create([
            'post_type' => 'event',
            'user_id' => $this->user->id,
            'event_id' => $event->id,
            'post_text' => 'Test post text',
            'post_category' => 'event',
            'status' => 1,
        ]);

        // Delete the event
        $response = $this->postJson('/api/event/delete', [
            'event_id' => $event->id,
        ]);

        $response->assertStatus(200);

        // Check that the post was deleted
        $this->assertDatabaseMissing('posts', [
            'id' => $post->id,
        ]);
    }

    /** @test */
    public function event_post_appears_in_newsfeed()
    {
        $this->actingAs($this->user);

        // Create an event
        $event = Event::factory()->create([
            'user_id' => $this->user->id,
            'name' => 'Newsfeed Test Event',
        ]);

        // Create the associated post
        $post = Post::create([
            'post_type' => 'event',
            'user_id' => $this->user->id,
            'event_id' => $event->id,
            'post_text' => 'Test event post for newsfeed',
            'post_category' => 'event',
            'privacy_level' => 1, // Public
            'status' => 1,
        ]);

        // Get newsfeed
        $response = $this->postJson('/api/post/newsfeed', [
            'limit' => 10,
            'offset' => 0,
        ]);

        $response->assertStatus(200);

        $data = $response->json('data');
        
        // Check that the event post is in the newsfeed
        $eventPost = collect($data)->firstWhere('post_type', 'event');
        $this->assertNotNull($eventPost);
        $this->assertEquals($post->id, $eventPost['id']);
        $this->assertEquals('event', $eventPost['post_type']);
        
        // Check that event details are included
        $this->assertArrayHasKey('event_details', $eventPost);
        $this->assertEquals($event->id, $eventPost['event_details']['id']);
        $this->assertEquals($event->name, $eventPost['event_details']['name']);
    }
} 